<?php

namespace App\Http\Controllers\Admin;

use Exception;
use App\Video;
use App\VideoDescription;
use Illuminate\View\View;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Routing\Redirector;
use App\Http\Requests\VideoRequest;
use App\Http\Controllers\Controller;
use Illuminate\Http\RedirectResponse;
use Mcamara\LaravelLocalization\Facades\LaravelLocalization;
use function GuzzleHttp\Promise\all;

class VideoController extends Controller
{
    /** Redirect to this path after each operation success*/
    private $redirectSuccessPath = '/admin/videos';
    /** View folder */
    private $viewDirectory = 'admin.videos.';

    public function __construct()
    {
        $this->redirectSuccessPath = getCurrentLocale().$this->redirectSuccessPath;
    }

    /**
     * Display a listing of the resource.
     * @param Request $request
     * @return View
     */
    public function index(Request $request) :View
    {
        $title= __('videos.head');
        $request= $request->toArray();
        return view($this->viewDirectory.'index', get_defined_vars());
    }

    /**
     * return data of the pages.
     *
     * @param Request $request
     * @return View
     */
    public function grid(Request $request) :View
    {
        $query = Video::latest()
            ->join('video_descriptions AS vd', 'videos.id', 'vd.video_id')
            ->join('languages', 'languages.id', 'vd.language_id')
            ->where('local', LaravelLocalization::getCurrentLocale())
            ->select(['vd.title', 'videos.*']);

        if ($request->date_from) {
            $query->whereDate('videos.created_at', '>=', $request->date_from);
        }

        if ($request->date_to) {
            $query->whereDate('videos.created_at', '<=', $request->date_to);
        }

        if ($request->name) {
            $query->where('vd.title', 'LIKE', '%'.$request->name.'%');
        }

        if ( !is_null($request->status) && $request->status == 0) {
            $query->onlyTrashed();
        }

        $videos= $query->paginate(100);

        return view($this->viewDirectory.'grid',compact('videos'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return View
     */
    public function create() :View
    {
        $action= route('videos.store');
        $head = metaFields('videos', 'new', getCurrentLocale()) ?? __('videos.new');
        $title = $head;

        return view( $this->viewDirectory.'form', get_defined_vars());
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param VideoRequest $request
     * @return RedirectResponse|Redirector
     */
    public function store(VideoRequest $request) :RedirectResponse
    {
        $data = $request->all();

        // update image to youtube api thumbnail if request have not image
        if ($request->video_url){
            $data['video'] = parseExternalVideoLink($request->video_url)['url'];
            $data['image'] = parseExternalVideoLink($request->video_url)['image'];
        }

        $video = Video::create($data);

        // Insert Description
        $this->saveData($request, $video->id);

        return redirect($this->redirectSuccessPath)->with('message', __('dashboard.saveDone'));
    }


    /**
     * Show the form for editing the specified resource.
     *
     * @param Video $video
     * @return View
     */
    public function edit(Video $video) :View
    {
        $action= route('videos.update', $video->id);
        $head = metaFields('videos', 'edit', getCurrentLocale()) ?? __('videos.edit');
        $title = $head;

        $query = VideoDescription::where('video_id', $video->id)
            ->join('languages', 'languages.id', 'video_descriptions.language_id')
            ->select(['video_descriptions.*', 'languages.local']);

        $galleryDescription= $query->get();

        foreach ($galleryDescription as $row) {
            $video[$row->local]= $row;
        }

        return view( $this->viewDirectory.'form', get_defined_vars());
    }

    /**
     * Update the specified resource in storage.
     *
     * @param VideoRequest $request
     * @param Video $video
     * @return RedirectResponse|Redirector
     */
    public function update(VideoRequest $request, Video $video) :RedirectResponse
    {
        $data = $request->all();

        if ($request->video_url){
            $data['video'] = parseExternalVideoLink($request->video_url)['url'];
            $data['image'] = parseExternalVideoLink($request->video_url)['image'];
        }

        // Update image if request has it
        $video->update($data);


        // Delete old description
        VideoDescription::where('video_id', $video->id)->delete();

        // Insert new description
        $this->saveData( $request, $video->id );

        return redirect($this->redirectSuccessPath)->with('message', __('dashboard.saveDone'));
    }

    /**
     * Handle Save form data
     *
     * @param VideoRequest $request
     * @param int $video_id
     * @return void
     */

    private function saveData(VideoRequest $request, int $video_id ):void
    {
        $requestData= $request->all();
        foreach (languages() as $lang) {
            $data=[
                'video_id'=> $video_id,
                'language_id'=> $lang->id,
                'title'=> $requestData['title_'.$lang->local],
            ];
            VideoDescription::create($data);
        }

    }


    /**
     * Remove the specified resource from storage.
     *
     * @param Request $request
     * @param int $id
     * @return JsonResponse
     * @throws Exception
     */
    public function destroy(Request $request, int $id) :JsonResponse
    {
        $video = Video::withTrashed()->find($id);
        if ($video) {
            if ($video->deleted_at) {
                $video->forceDelete();
            } else {
                $video->delete();
            }
            return response()->json(['message'=> __('dashboard.deletedDone')]);
        } else{
            return response()->json(['message'=> __('dashboard.noResult')], 400);
        }
    }

    /**
     * Remove several pages by IDs.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function destroyAll(Request $request) :JsonResponse
    {
        $ids= $request->ids;
        if ($request->force) {
            Video::onlyTrashed()->whereIn('id', $ids)->forceDelete();
        } else {
            Video::whereIn('id', $ids)->delete();
        }

        return response()->json(['message'=> __('dashboard.deletedDone')]);
    }

    /**
     * Restore the specified category from storage
     *
     * @param Request $request
     * @param int $id
     * @return JsonResponse
     */
    public function restore(Request $request, int $id) :JsonResponse
    {
        $video = Video::withTrashed()->find($id);

        if ($video){
            $video->restore();
            return response()->json(['message'=> __('dashboard.saveDone')]);
        }
        return response()->json(['message'=> __('dashboard.noResult')], 400);
    }

    /**
     * Restore several pages by IDs.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function restoreAll(Request $request) :JsonResponse
    {
        $video = Video::whereIn('id', $request->ids)->onlyTrashed();
        if ($video){
            $video->restore();
            return response()->json(['message'=> __('dashboard.saveDone')]);
        }
    }
}
