<?php

namespace App\Http\Requests;

use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Request;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\ValidationException;
use Illuminate\Contracts\Validation\Validator;
use Illuminate\Http\Exceptions\HttpResponseException;

/**
 * @property mixed video_url
 * @property mixed type
 */
class VideoRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize()
    {
        if (! Auth::check()) {
            return false;
        }
        if(in_array(Auth::user()->role, ['super_admin', 'admin', 'sub_admin']))
        {
            return true;
        }

        if(Auth::user()->hasModule('videos'))
        {
            return true;
        }
        return Auth::check();
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules()
    {
        $rules=[
            'video' => 'sometimes|required_if:type,2',
            'image' => 'sometimes|required_if:type,2',
            'video_url' => 'sometimes|required_if:type,1',
            'type' => 'sometimes',
            'video_category_id' => 'required',
        ];


        foreach (languages() as $lang) {
            $lang_rules=[
                'title_'.$lang->local=> 'required|min:3',
            ];
            $rules= array_merge($rules, $lang_rules);
        }

        return $rules;
    }

    /**
     * Get custom attributes for validator errors.
     *
     * @return array
     */
    public function attributes()
    {
        $field = metaFields('videos');
        $local = getCurrentLocale();
        $attributes = [
            'video_category_id' => $field[$local]['video_category_id'] ??  __('videos.category'),
            'video' => $field[$local]['video'] ??  __('videos.category'),
            'video_url' => $field[$local]['video_url'] ??  __('videos.video_url_'),
            'image' => $field[$local]['image'] ??  __('videos.image'),
            'type' => $field[$local]['type'] ??  __('videos.type'),
        ];
        foreach (languages() as $lang) {
            $lang_attributes=[
                'title_'.$lang->local=> $field[$lang->local]['title'] ??  __('videos.name'),
            ];
            $attributes= array_merge($attributes, $lang_attributes);
        }
        return $attributes;
    }


    /**
     * Handle a failed validation attempt.
     *
     * Override the parent method action if the request is for AJAX
     *
     * @param Validator $validator
     * @return void
     *
     * @throws ValidationException
     */
    protected function failedValidation(Validator $validator)
    {
        // Change response attitude if the request done via Ajax requests like API requests
        if(Request::wantsJson()){
            $errors = (new ValidationException($validator))->errors();
            throw new HttpResponseException(response()->json(['errors' => $errors], 422));
        } else {
            // Do the original action of the Form request class
            parent::failedValidation($validator);
        }
    }
}
