<?php

namespace App\Http\Requests;

use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Request;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Contracts\Validation\Validator;
use Illuminate\Validation\ValidationException;
use Illuminate\Http\Exceptions\HttpResponseException;

/**
 * @property mixed images
 */
class WebsiteCategoryRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize()
    {
        if (! Auth::check()) {
            return false;
        }
        if(in_array(Auth::user()->role, ['super_admin', 'admin', 'sub_admin']))
        {
            return true;
        }

        if(Auth::user()->hasModule('websiteCategories'))
        {
            return true;
        }
        return Auth::check();
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules()
    {
        $local = getCurrentLocale();
        $rules=[
            'name_'.$local => 'required|min:3',
            'slug_'.$local=> [
                "required",
                'min:3',
                Rule::unique('website_category_descriptions', 'slug')->ignore($this->websiteCategory?$this->websiteCategory->id:0, 'website_category_id')
            ],
            'images' => 'required',
            'image_title_'.$local.'.*' => 'required|min:3',
            'image_url.*' => 'required|url',
        ];

        return $rules;
    }

    /**
     * Get custom attributes for validator errors.
     *
     * @return array
     */
    public function attributes()
    {
        $field = metaFields('websiteCategories');
        $local = getCurrentLocale();
        $attributes=[
            'name_'.$local => $field[$local]['name'] ?? __('websiteCategories.name'),
            'slug_'.$local => $field[$local]['slug'] ?? __('websiteCategories.slug'),
            'images' => $field[$local]['images'],
        ];
        if (request()->has('images')){
            foreach (request('images') as $key => $image) {
                $imagesAttributes=[
                    'image_title_'.$local.'.'.$key=> $field[$local]['image_title'] ?? __('websiteCategories.title'),
                    'image_url'.'.'.$key=> $field[$local]['image_url'] ?? __('websiteCategories.url'),
                ];

                $attributes= array_merge($attributes, $imagesAttributes);
            }
        }


        return $attributes;
    }
    /**
     * Handle a failed validation attempt.
     *
     * Override the parent method action if the request is for AJAX
     *
     * @param Validator $validator
     * @return void
     *
     * @throws ValidationException
     */
    protected function failedValidation(Validator $validator)
    {
        // Change response attitude if the request done via Ajax requests like API requests
        if(Request::wantsJson()){
            $errors = (new ValidationException($validator))->errors();
            throw new HttpResponseException(response()->json(['errors' => $errors], 422));
        } else {
            // Do the original action of the Form request class
            parent::failedValidation($validator);
        }
    }
}
