<?php

namespace App\Http\Requests;

use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Request;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\ValidationException;
use Illuminate\Contracts\Validation\Validator;
use Illuminate\Http\Exceptions\HttpResponseException;

/**
 * @property mixed video_type
 * @property mixed video_url
 * @property mixed images
 */
class WorkHourRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize()
    {
        if (! Auth::check()) {
            return false;
        }
        if(in_array(Auth::user()->role, ['super_admin', 'admin', 'sub_admin']))
        {
            return true;
        }

        if(Auth::user()->hasModule('workHours'))
        {
            return true;
        }
        return Auth::check();
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules()
    {
        $rules = [
            'week_day' => [
                'required',
                'in:sat,sun,mon,tue,wed,thu,fri',
                Rule::unique('work_hours')->ignore($this->workHour ? $this->workHour->week_day : 0, 'week_day')
            ],

            'status' => 'required|in:open,closed',
            'time_from' => 'required_if:status,open',
            'time_to' => 'required_if:status,open',
        ];

        return $rules;
    }

    /**
     * Get custom attributes for validator errors.
     *
     * @return array
     */
    public function attributes()
    {
        $field = metaFields('workHours');
        $local = getCurrentLocale();
        $attributes=[
            'week_day' =>  $field[$local]['week_day'] ?? __('workHours.week_day'),
            'status' =>  $field[$local]['status'] ?? __('workHours.status'),
            'time_from' =>  $field[$local]['time_from'] ?? __('workHours.time_from'),
            'time_fto' =>  $field[$local]['time_fto'] ?? __('workHours.time_fto'),
        ];

        return $attributes;
    }

    public function messages()
    {
        return [
            'time_from.required_if' => __('workHours.time_from_required_if'),
            'time_to.required_if' => __('workHours.time_to_required_if'),
        ];
    }


    /**
     * Handle a failed validation attempt.
     *
     * Override the parent method action if the request is for AJAX
     *
     * @param Validator $validator
     * @return void
     *
     * @throws ValidationException
     */
    protected function failedValidation(Validator $validator)
    {
        // Change response attitude if the request done via Ajax requests like API requests
        if(Request::wantsJson()){
            $errors = (new ValidationException($validator))->errors();
            throw new HttpResponseException(response()->json(['errors' => $errors], 422));
        } elseif (Request::ajax()){
            $errors = $validator->errors()->first();
            throw new HttpResponseException(response()->json(['errors' => $errors]));
        } else {
            parent::failedValidation($validator);
        }
    }
}
