<?php

namespace App\Http\Requests;

use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Request;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\ValidationException;
use Illuminate\Contracts\Validation\Validator;
use Illuminate\Http\Exceptions\HttpResponseException;

/**
 * @property mixed type
 * @property mixed video_url
 * @property mixed image
 * @property mixed file
 */
class GalleryRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize()
    {
        if (! Auth::check()) {
            return false;
        }
        if(in_array(Auth::user()->role, ['super_admin', 'admin', 'sub_admin']))
        {
            return true;
        }

        if(Auth::user()->hasModule('galleries'))
        {
            return true;
        }
        return Auth::check();
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules()
    {

        $rules = [
            'type' => 'required|in:image_type,video_type,magazine_type',
            'video_url' => 'required_if:type,video_type|url',
        ];

        $linkPattern = '/^(https:\/\/(www\.)?(youtu|youtube){1}.([a-z]{1,4}))(\/watch\?v=|\/embed\/|\/){0,1}([a-zA-z0-9]{2,15})(&.)*/';
        $videoUrl = request('video_url');
        preg_match($linkPattern, $videoUrl, $linkMatches);

        if (!count($linkMatches)){
            $rules['video_url_validate'] = 'required';
        }

        foreach (languages() as $lang) {
            $lang_rules=[
                "title_$lang->local" => 'required|min:3|max:100',
            ];
            $rules= array_merge($rules, $lang_rules);
        }

        $allowedFiles = ['.pdf', '.jpg', '.jpeg', '.png'];

        $image = request('image');

        $isValid = false;

        foreach ($allowedFiles as $allowedFile) {
            if (strpos($image, $allowedFile)){
                $isValid = true;
                break;
            }
        }

        if (!$isValid){
            $rules['image_validate'] = 'required';
        }

        $rules['image'] = 'required';

        return $rules;
    }

    /**
     * Get custom attributes for validator errors.
     *
     * @return array
     */
    public function attributes()
    {
        $field = metaFields('galleries');
        $local = getCurrentLocale();
        $attributes=[
            'type' => $field[$local]['type'] ?? __('galleries.type'),
            'video_url' => $field[$local]['video_url'] ?? __('galleries.video_url'),
            'video_url_validate' => $field[$local]['video_url'] ?? __('galleries.video_url'),
            'image' => $field[$local]['image'] ?? __('galleries.image'),
            'image_validate' => $field[$local]['image'] ?? __('galleries.image'),
            'file' => $field[$local]['file'] ?? __('galleries.file'),
        ];


        foreach (languages() as $lang) {
            $local = $lang->local == 'ar' ? ' (عربي) ' : ' (English) ';
            $lang_attributes=[
                'title_'.$lang->local => ($field[$lang->local]['title'] ?? __('galleries.title')) . $local,
            ];
            $attributes= array_merge($attributes, $lang_attributes);
        }
        return $attributes;
    }


    public function messages()
    {
        $attributes = [
            'video_url.required_if' => __('galleries.video_url_required'),
            'video_url_validate.required' => __('galleries.video_url_validate'),
            'image_validate.required' => __('galleries.image_file_types'),
        ];
        return $attributes;
    }


    /**
     * Handle a failed validation attempt.
     *
     * Override the parent method action if the request is for AJAX
     *
     * @param Validator $validator
     * @return void
     *
     * @throws ValidationException
     */
    protected function failedValidation(Validator $validator)
    {
        // Change response attitude if the request done via Ajax requests like API requests
        if(Request::ajax()){
            if ($validator->fails()){
                $errors = $validator->errors()->first();
                throw new HttpResponseException(response()->json(['errors' => $errors]));
            }

        } elseif (Request::ajax()){
            $errors = $validator->errors()->first();
            throw new HttpResponseException(response()->json(['errors' => $errors]));
        } else {
            // Do the original action of the Form request class
            parent::failedValidation($validator);
        }
    }
}
