<?php

namespace App\Http\Requests;

use Illuminate\Support\Facades\Request;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\ValidationException;
use Illuminate\Contracts\Validation\Validator;
use Illuminate\Http\Exceptions\HttpResponseException;

/**
 * @property mixed video_type
 * @property mixed video_url
 * @property mixed images
 */
class CareerApplyRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize()
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules()
    {
        $phonePattern = '/^(009665|9665|\+9665|05|5)(5|0|3|6|4|9|1|8|7)([0-9]{7})$/';

        $rules = [
            'name' => 'required|min:3|max:100',
            'phone' => ['required','numeric','regex:'.$phonePattern],
            'country_id' => 'required|exists:countries,id',
            'address' => 'required|min:3|max:100',
            'birth_date' => 'required|date',
            'education_id' => 'required|exists:educations,id',
//            'service_id' => 'required|exists:services,id',
            'cv' => 'required|mimes:pdf',
        ];

        return $rules;
    }

    /**
     * Get custom attributes for validator errors.
     *
     * @return array
     */
    public function attributes()
    {
        $field = metaFields('careerApplications');
        $local = getCurrentLocale();
        $attributes=[
            'name' =>  $field[$local]['name'] ?? __('careerApplications.name'),
            'phone' =>  $field[$local]['phone'] ?? __('careerApplications.phone'),
            'country_id' =>  $field[$local]['country_id'] ?? __('careerApplications.country_id'),
            'address' =>  $field[$local]['address'] ?? __('careerApplications.address'),
            'birth_date' =>  $field[$local]['birth_date'] ?? __('careerApplications.birth_date'),
            'education_id' =>  $field[$local]['education_id'] ?? __('careerApplications.education_id'),
//            'service_id' =>  $field[$local]['service_id'] ?? __('careerApplications.service_id'),
            'cv' =>  $field[$local]['cv'] ?? __('careerApplications.cv'),
        ];

        return $attributes;
    }


    /**
     * Handle a failed validation attempt.
     *
     * Override the parent method action if the request is for AJAX
     *
     * @param Validator $validator
     * @return void
     *
     * @throws ValidationException
     */
    protected function failedValidation(Validator $validator)
    {
        // Change response attitude if the request done via Ajax requests like API requests
        if(Request::wantsJson()){
            $errors = (new ValidationException($validator))->errors();
            throw new HttpResponseException(response()->json(['errors' => $errors], 422));
        } elseif (Request::ajax()){
            $errors = $validator->errors()->first();
            throw new HttpResponseException(response()->json(['errors' => $errors]));
        } else {
            parent::failedValidation($validator);
        }
    }
}
