<?php

namespace App\Http\Controllers\Admin;

use App\Address;
use Exception;
use App\Language;
use App\AddressDescription;
use Illuminate\View\View;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use App\Http\Requests\AddressRequest;
use Illuminate\Routing\Redirector;
use App\Http\Controllers\Controller;
use Illuminate\Http\RedirectResponse;
use Mcamara\LaravelLocalization\Facades\LaravelLocalization;

class AddressController extends Controller
{

    /** Redirect to this path after each operation success*/
    private $redirectSuccessPath = '/admin/addresses';
    /** View folder */
    private $viewDirectory = 'admin.addresses.';

    public function __construct()
    {
        $this->redirectSuccessPath = getCurrentLocale().$this->redirectSuccessPath;
    }

    /**
     * Display a listing of the resource.
     * @param Request $request
     * @return View
     */
    public function index(Request $request) :View
    {
        $title= __('addresses.head');
        $request= $request->toArray();
        return view($this->viewDirectory.'index', get_defined_vars());
    }

    /**
     * return data of the addresses.
     *
     * @param Request $request
     * @return View
     */
    public function grid(Request $request) :View
    {
        $query = Address::latest()
            ->join('address_descriptions AS aDesc', 'addresses.id', 'aDesc.address_id')
            ->where('aDesc.language_id', currentLanguage()->id)
            ->select(['aDesc.name', 'addresses.*']);

        if ($request->date_from) {
            $query->whereDate('addresses.created_at', '>=', $request->date_from);
        }

        if ($request->date_to) {
            $query->whereDate('addresses.created_at', '<=', $request->date_to);
        }

        if ($request->name) {
            $query->where('aDesc.name', 'LIKE', '%'.$request->name.'%');
        }

        if ( !is_null($request->status) && $request->status == 0) {
            $query->onlyTrashed();
        }

        $addresses= $query->paginate(100);

        return view($this->viewDirectory.'grid', get_defined_vars());
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return View
     */
    public function create() :View
    {
        $action= route('addresses.store');
        $head = metaFields('addresses', 'new', getCurrentLocale()) ?? __('addresses.new');
        $title = $head;

        return view( $this->viewDirectory.'form', get_defined_vars());
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param AddressRequest $request
     * @return JsonResponse
     */
    public function store(AddressRequest $request) :JsonResponse
    {
        $address = Address::create($request->all());
        // Insert Description
        $this->saveData( $request, $address->id );
        session()->flash('message', __('dashboard.saveDone'));
        return response()->json(['url' => route('addresses.index')]);
//        return redirect($this->redirectSuccessPath)->with('message', __('dashboard.saveDone'));
    }


    /**
     * Show the form for editing the specified resource.
     *
     * @param Address $address
     * @return View
     */
    public function edit(Address $address) :View
    {
        $action= route('addresses.update', $address->id);
        $head = metaFields('addresses', 'edit', getCurrentLocale()) ?? __('addresses.edit');
        $title = $head;

        $query = AddressDescription::where('address_id', $address->id)
            ->join('languages', 'languages.id', 'address_descriptions.language_id')
            ->select(['address_descriptions.*', 'languages.local']);

        $addressDescription= $query->get();

        foreach ($addressDescription as $row) {
            $address[$row->local]= $row;
        }

        return view( $this->viewDirectory.'form', get_defined_vars());
    }

    /**
     * Update the specified resource in storage.
     *
     * @param AddressRequest $request
     * @param Address $address
     * @return JsonResponse
     */
    public function update(AddressRequest $request, Address $address) : JsonResponse
    {
        // Update image if request has it
        $address->update($request->all());

        // Delete old description
        AddressDescription::where('address_id', $address->id)->delete();

        // Insert new description
        $this->saveData( $request, $address->id );
        session()->flash('message', __('dashboard.saveDone'));
        return response()->json(['url' => route('addresses.index')]);
//        return redirect($this->redirectSuccessPath)->with('message', __('dashboard.saveDone'));
    }

    /**
     * Handle Save form data
     *
     * @param AddressRequest $request
     * @param int $address_id
     * @return void
     */

    private function saveData(AddressRequest $request, int $address_id ):void
    {
        $requestData= $request->all();
        $languages= Language::where('status', 1)->cursor();

        foreach ($languages as $lang) {
            $data=[
                'address_id'=> $address_id,
                'language_id'=> $lang->id,
                'name'=> $requestData['name_'.$lang->local],
            ];
            AddressDescription::create($data);
        }

    }


    /**
     * Remove the specified resource from storage.
     *
     * @param Request $request
     * @param int $id
     * @return JsonResponse
     * @throws Exception
     */
    public function destroy(Request $request, int $id) :JsonResponse
    {
        if (1 == $id) {
            return response()->json(['message'=> __('dashboard.noResult')], 400);
        }

        $address = Address::withTrashed()->find($id);
        if ($address) {
            if ($address->deleted_at) {
                $address->forceDelete();
            } else {
                $address->delete();
            }
            return response()->json(['message'=> __('dashboard.deletedDone')]);
        } else{
            return response()->json(['message'=> __('dashboard.noResult')], 400);
        }
    }

    /**
     * Remove several addresses by IDs.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function destroyAll(Request $request) :JsonResponse
    {
        $ids= $request->ids;
        if ( in_array( 1, $ids) ) {
            return response()->json(['message'=> __('dashboard.noResult')], 400);
        }

        if ($request->force) {
            Address::onlyTrashed()->whereIn('id', $ids)->forceDelete();
        } else {
            Address::whereIn('id', $ids)->delete();
        }

        return response()->json(['message'=> __('dashboard.deletedDone')]);
    }

    /**
     * Restore the specified category from storage
     *
     * @param Request $request
     * @param int $id
     * @return JsonResponse
     */
    public function restore(Request $request, int $id) :JsonResponse
    {
        $address = Address::withTrashed()->find($id);

        if ($address){
            $address->restore();
            return response()->json(['message'=> __('dashboard.saveDone')]);
        }
        return response()->json(['message'=> __('dashboard.noResult')], 400);
    }

    /**
     * Restore several addresses by IDs.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function restoreAll(Request $request) :JsonResponse
    {
        $addresses = Address::whereIn('id', $request->ids)->onlyTrashed();
        if ($addresses){
            $addresses->restore();
            return response()->json(['message'=> __('dashboard.saveDone')]);
        }
    }
}
