<?php

namespace App\Http\Controllers\Admin;

use App\Country;
use App\Http\Requests\CountryRequest;
use Exception;
use App\CountryDescription;
use Illuminate\View\View;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use App\Http\Controllers\Controller;

class CountryController extends Controller
{

    /** Redirect to this path after each operation success*/
    private $redirectSuccessPath = '/admin/countries';
    /** View folder */
    private $viewDirectory = 'admin.countries.';

    public function __construct()
    {
        $this->redirectSuccessPath = getCurrentLocale().$this->redirectSuccessPath;
    }


    /**
     * Display a listing of the resource.
     * @param Request $request
     * @return View
     */
    public function index(Request $request) :View
    {
        $title= __('countries.head');
        $request= $request->toArray();
        return view($this->viewDirectory.'index', get_defined_vars());
    }

    /**
     * return data of the courses.
     *
     * @param Request $request
     * @return View
     */
    public function grid(Request $request) :View
    {
        $query = Country::join('country_descriptions AS cd', 'countries.id', 'cd.country_id')
            ->where('cd.language_id', currentLanguage()->id)
            ->select(['cd.name', 'countries.*']);

        if ($request->date_from) {
            $query->whereDate('countries.created_at', '>=', $request->date_from);
        }

        if ($request->date_to) {
            $query->whereDate('countries.created_at', '<=', $request->date_to);
        }

        if ($request->name) {
            $query->where('cd.name', 'LIKE', '%'.$request->name.'%');
        }
        if ($request->code) {
            $query->where('countries.code', 'LIKE', '%'.$request->code.'%');
        }


        if ( !is_null($request->status) && $request->status == 0) {
            $query->onlyTrashed();
        }

        $countries= $query->paginate(100);

        return view($this->viewDirectory.'grid',get_defined_vars());
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return View
     */
    public function create() :View
    {
        $action= route('countries.store');
        $head = metaFields('countries', 'add_new', getCurrentLocale()) ?? __('countries.new');
        $local = getCurrentLocale();
        return view( $this->viewDirectory.'form', get_defined_vars());
    }

    /**
     * Store a newly created resource in storage.
     * @param CountryRequest $request
     * @return JsonResponse
     */
    public function store(CountryRequest $request) :JsonResponse
    {
        $country = Country::create($request->all());
        $this->saveData( $request, $country->id );
        session()->flash('message', __('dashboard.saveDone'));
        return response()->json(['url' => route('countries.index')]);
//        return redirect($this->redirectSuccessPath)->with('message', __('dashboard.saveDone'));
    }


    /**
     * Show the form for editing the specified resource.
     *
     * @param Country $country
     * @return View
     */
    public function edit(Country $country) :View
    {
        $action= route('countries.update', $country->id);
        $head = metaFields('countries', 'edit', getCurrentLocale()) ?? __('countries.edit');
        $title = $head;
        $query = CountryDescription::where('country_id', $country->id)
            ->join('languages', 'languages.id', 'country_descriptions.language_id')
            ->select(['country_descriptions.*', 'languages.local']);

        $countryDescription= $query->get();

        foreach ($countryDescription as $row) {
            $country[$row->local]= $row;
        }

        $local = getCurrentLocale();

        return view( $this->viewDirectory.'form', get_defined_vars());
    }

    /**
     * Update the specified resource in storage.
     * @param CountryRequest $request
     * @param Country $country
     * @return JsonResponse
     */
    public function update(CountryRequest $request, Country $country) :JsonResponse
    {
        $country->update($request->all());
        // Delete old description
        CountryDescription::where('country_id', $country->id)->delete();
        $this->saveData( $request, $country->id );
        session()->flash('message', __('dashboard.saveDone'));
        return response()->json(['url' => route('countries.index')]);
//        return redirect($this->redirectSuccessPath)->with('message', __('dashboard.saveDone'));
    }


    /**
     * Handle Save form data
     *
     * @param CountryRequest $request
     * @param int $country_id
     * @return void
     */

    private function saveData(CountryRequest $request, int $country_id ):void
    {
        $requestData= $request->all();
        foreach (languages() as $lang) {
            $data=[
                'name'=> $requestData['name_'.$lang->local],
                'country_id'=> $country_id,
                'language_id'=> $lang->id,
            ];
            CountryDescription::create($data);
        }

    }



    /**
     * Remove the specified resource from storage.
     *
     * @param Request $request
     * @param int $id
     * @return JsonResponse
     * @throws Exception
     */
    public function destroy(Request $request, int $id) :JsonResponse
    {
        $country = Country::withTrashed()->find($id);
        if ($country) {
            if ($country->deleted_at) {
                $country->forceDelete();
            } else {
                $country->delete();
            }
            return response()->json(['message'=> __('dashboard.deletedDone')]);
        } else{
            return response()->json(['message'=> __('dashboard.noResult')], 400);
        }
    }

    /**
     * Remove several courses by IDs.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function destroyAll(Request $request) :JsonResponse
    {
        $ids= $request->ids;
        if ($request->force) {
            Country::onlyTrashed()->whereIn('id', $ids)->forceDelete();
        } else {
            Country::whereIn('id', $ids)->delete();
        }

        return response()->json(['message'=> __('dashboard.deletedDone')]);
    }

    /**
     * Restore the specified category from storage
     *
     * @param Request $request
     * @param int $id
     * @return JsonResponse
     */
    public function restore(Request $request, int $id) :JsonResponse
    {
        $country = Country::withTrashed()->find($id);

        if ($country){
            $country->restore();
            return response()->json(['message'=> __('dashboard.saveDone')]);
        }
        return response()->json(['message'=> __('dashboard.noResult')], 400);
    }

    /**
     * Restore several courses by IDs.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function restoreAll(Request $request) :JsonResponse
    {
        $countries = Country::whereIn('id', $request->ids)->onlyTrashed();
        if ($countries){
            $countries->restore();
            return response()->json(['message'=> __('dashboard.saveDone')]);
        }
    }
}
